function [S,E,I,R]=seir_sim(beta,eta,gamma,E0,I0,V,target)
%
% function results=seir_sim(beta,eta,gamma,E0,I0,V)
%
% runs a simulation of an SEIR model
%
% S: susceptible
% E: 'exposed' ('latent' is the more accurate term)
% I: infective
% R: removed
%
% beta is the infectivity parameter, scaled so that N=1
% eta is the reciprocal of mean incubation time
% gamma is the reciprocal of mean recovery time
% E0 is the initial latent fraction
% I0 is the initial infectious fraction
% V is the initial immune fraction
% target is the infected fraction used as the end condition
%
% by Glenn Ledder
% written 2020/08/14
% revised 2020/10/27
%
% direct comments to gledder@unl.edu

%% DATA

% suggested default values
% beta = 0.25;
% eta = 0.2;
% gamma = 0.1;
% E0 = 0;
% I0 = 0.01;
% V = 0;
% target = 0.001;

maxdays = 1000;

%% INITIALIZATION

% set up results data structure with Y=[S,E,I] and R

results = zeros(maxdays+1,4);
S0 = 1-E0-I0-V;
Y = [S0,E0,I0];
R = V;
results(1,:) = [Y,R];

y = Y';
oldi = I0;

%% COMPUTATION

for t=1:maxdays
    % y is a column vector, Y^T
    y = rk4(1,y);
    Y = y';
    R = 1-sum(Y);
    results(t+1,:) = [Y,R];
    if Y(2)+Y(3)>min(target,oldi)
        oldi = Y(2)+Y(3);
    else
        results = results(1:(t+1),:);
        break;
    end
end

S = results(:,1);
E = results(:,2);
I = results(:,3);
R = results(:,4);

%% FUNCTION FOR rk4

    function y=rk4(dt,y0)
        % y0 is a column vector of initial conditions at t
        % y is a column vector of values at t+dt
        k1 = yprime(y0);
        k2 = yprime(y0+0.5*dt*k1);
        k3 = yprime(y0+0.5*dt*k2);
        k4 = yprime(y0+dt*k3);
        y = y0+dt*(k1+2*k2+2*k3+k4)/6;
    end
        
%% FUNCTION FOR THE DIFFERENTIAL EQUATION

    function yp=yprime(y)
    % split out components
        S = y(1);
        E = y(2);
        I = y(3);
    % compute derivatives
        Sp = -beta*S*I;
        Ep = -Sp-eta*E;
        Ip = eta*E-gamma*I;
    % assemble derivative
        yp = [Sp;Ep;Ip];
    end

%% END

end
